;------------------------------------------------------------------------
; Assemble command
;------------------------------------------------------------------------

CMD_ASM         JSR     FIND_EOFZ       Find end of file
                LDY     PNTR            Set begin of symbol table
                LDX     PNTR+1
                INY                     Symtab starts 1 byte after EOF
                BNE     .NOCY           No carry!
                INX
.NOCY           STY     SYM_TABLE
                STX     SYM_TABLE+1

                LDY     #0              I presume that we're not beyond
                TYA                      himem already for memory's sake
                STA     (SYM_TABLE),Y   Clear symbol table
                STA     PASS            Set pass 0
                STA     ASM_ERR         Clear error counter

                JSR     ASSEMBLE        Start pass 1
                LDA     ASM_ERR         Were there any errors?
                BNE     .ERRORS         Yep!
                DEC     PASS            Start pass 2 (Must be $FF !)
                JSR     ASSEMBLE
                JSR     PRINT_OBJ       Print last object range

.ERRORS         LDA     ASM_ERR         Print number of errors
                STA     HEXVAL

                .DO     PROCESSOR=$6502
                LDA     #0
                STA     HEXVAL+1
                .EL
                STZ     HEXVAL+1
                .FI

                JSR     PRDECI
                LDY     #STR_ERRORS     Print errors string behind it
                JMP     PRINT

;------------------------------------------------------------------------
; This is what we came for, assembling!!!!
;------------------------------------------------------------------------

ASSEMBLE        JSR     SET_BOF         Start at begin of program

                .DO     PROCESSOR=$6502
                LDA     #0
                STA     GLOBAL+1        Make last global label invalid
                .EL
                STZ     GLOBAL+1        Make last global label invalid
                .FI

                LDA     #DEF_ORG        Setup default PC
                STA     PC
                STA     TA
                STA     TA_BEGIN
                LDA     /DEF_ORG
                STA     PC+1
                STA     TA+1
                STA     TA_BEGIN+1

.LINE           JSR     HALT            Halt output if necessary
                BEQ     .DONE           ESC was pressed!
                LDA     PC              Copy PC at begin of line
                STA     PC_BEG
                LDA     PC+1
                STA     PC_BEG+1
                LDY     #0              End of program?
                LDA     (PNTR),Y
                BEQ     .DONE           Yes! We're done (with this pass)
                STY     UNDEF           Clear undefined label flag
                STY     FORWARD          and forward ref. label flag
                STY     ERROR           Clear error number
                JSR     UNPACK          Unpack this line to IN

                LDX     #0
                LDA     IN              Can it be a label?
                CMP     #" "
                BEQ     .NO_LABEL       Nope!
                CMP     #CR             Can't be CR, but just in case
                BEQ     .NEXT           Do next line!
                CMP     #";"
                BEQ     .NEXT           It's a comment, ignore line
                JSR     ADD_LABEL       Add this label to symbol table

.NO_LABEL       JSR     NNONSPC         Find next non space now
                BEQ     .NEXT           That was all, next line please!
                CMP     #";"
                BEQ     .NEXT           The rest is comment!
                CMP     #"."            Can it be a directive?
                BEQ     .DIRECT         Yes! Handle it
                JSR     MNEMON          Decode mnemonic and operand
                JMP     .NEXT           That's it!

.DIRECT         JSR     DIRECT          Decode directive

.NEXT           JSR     CHCK_ERROR      Handle errors
                JSR     NEXT_LINE       Point to next line
                BNE     .LINE           Not EOF!

.DONE           RTS

;------------------------------------------------------------------------
; Check error
; If ERROR<>0 print line number and error behind it
; Also increment error counter
;------------------------------------------------------------------------

CHCK_ERROR      LDA     ERROR           Did an error occur?
                BNE     .ERRORS         Yes!
                LDA     UNDEF           Undefined label used in pass 2?
                AND     PASS             (0 if pass 1)
                BEQ     .RTS            Nope!
                JSR     UNDEF_ERROR     Exit with undefined label error

.ERRORS         LDY     #1              Print line number
                LDA     (PNTR),Y
                STA     HEXVAL
                INY
                LDA     (PNTR),Y
                STA     HEXVAL+1
                JSR     PRDECI
                JSR     PRSPACE
                JSR     PRINT_ERROR     Print the error
                INC     ASM_ERR         Increment error counter
                BNE     .RTS            No error overflow!
                DEC     ASM_ERR         Max indicator is 255 errors.
.RTS            RTS

;------------------------------------------------------------------------
; Decode directive
;------------------------------------------------------------------------

DIRECT          JSR     NEXT_IN         Get directive's name
                BEQ     .ERROR          It may not be CR already!
                STA     PARM1
                JSR     NEXT_IN_Y0
                BEQ     .ERROR          It may not be CR already!
                STA     PARM1+1
.SEARCH         LDA     .TABLE,Y        Could it be this one?
                BEQ     .ERROR          Nope! Directive not found!
                CMP     PARM1
                BNE     .NEXT           Nope!
                LDA     .TABLE+1,Y
                CMP     PARM1+1
                BEQ     .GOTIT          Found directive!
.NEXT           INY                     Point to next directive in table
                INY
                BNE     .SEARCH         Always taken!

.GOTIT          LDA     .JUMPS+1,Y      Get directive handler's address
                PHA
                LDA     .JUMPS,Y
                PHA

.EOF_DIR        JSR     NEXT_IN_Y0      Find end of directive first
                BEQ     .JUMP           Execute directive!
                CMP     #" "            Is it a space already?
                BNE     .EOF_DIR        Nope, not end of directive yet!
                JMP     NNONSPC         Find next non space

.ERROR          LDA     #ERR_DIR        Exit with directive error
                STA     ERROR
.JUMP           RTS

.TABLE          .AS     -/AS/           ASCII string
                .AS     -/AT/           ASCII string terminated
                .AS     -/BS/           Block skip
                .AS     -/DA/           Data directive
                .AS     -/EQ/           EQ directive
                .AS     -/OR/           ORG directive
                .AS     -/TA/           Target address directive
                .DA     #0              End of table

.JUMPS          .DA     DIR_AS-1        ASCII string
                .DA     DIR_AT-1        ASCII string terminated
                .DA     DIR_BS-1        Block skip
                .DA     DIR_DA-1        Data directive
                .DA     DIR_EQ-1        EQ directive
                .DA     DIR_OR-1        ORG directive
                .DA     DIR_TA-1        Target address directive

;------------------------------------------------------------------------
; .AS and .AT directives
;------------------------------------------------------------------------

DIR_AT          LDY     #%1000.0000     Set end flag
DIR_AS          STY     LEAD0           Save end flag

                LDY     #%0111.1111     In case of positive ASCII
                CMP     #"-"            Is it the sign flag?
                BNE     .POS            No! Positive ASCII
                LDY     #%1111.1111     Set negative ASCII flag
                JSR     NEXT_IN         Point to next character
                BEQ     DIR_OPE         Premature EOL!

.POS            STY     CHAR            Save ASCII polarity
                STA     DELIM           Save delimiter character

.LOOP           JSR     NEXT_IN         Get next char
                BEQ     DIR_OPE         Sudden EOL!
                CMP     DELIM           End of string?
                BEQ     .EXIT           Yes!
                AND     CHAR            Set/clear b7
                LDY     IN+1,X          Is next char the delimiter?
                CPY     DELIM
                BNE     .NOT_END        Nope!
                EOR     LEAD0           Invert b7 if end flag set
.NOT_END        JSR     SAVE_OBJ        Save byte to object
                JMP     .LOOP           Do all characters

.EXIT           JSR     NEXT_IN         Next char must be blank or EOL
                BEQ     .RTS            It is!
                CMP     #" "
                BNE     DIR_OPE         It is not!
.RTS            RTS

;------------------------------------------------------------------------
; .BS directive
;------------------------------------------------------------------------

DIR_BS          JSR     GET_EXPRES      Evaluate expression
                BNE     DIR_EXIT        An error occurred
                LDA     UNDEF
                BNE     .UNDEF          Undefined labels not allowed
                LDA     PASS            Don't increment TA during pass 1
                BEQ     .PC
                LDY     #TA             Add expression to TA
                JSR     .ADD

.PC             LDY     #PC             Add expression to PC
.ADD            CLC
                LDA     0,Y
                ADC     HEXVAL
                STA     0,Y
                LDA     1,Y
                ADC     HEXVAL+1
                STA     1,Y
                RTS

.UNDEF          JMP     DIR_UNDEF       Undefined label used!

DIR_OPE         LDA     #ERR_OPE        Exit with operand error
                STA     ERROR
DIR_EXIT        RTS

;------------------------------------------------------------------------
; .DA directive
;------------------------------------------------------------------------

DIR_DA          STA     CHAR            Save prefix
                CMP     #"#"            Low byte only?
                BEQ     .PREFIX         Yes!
                CMP     #"/"            High byte only?
                BNE     .NOPREFIX       Nope!
.PREFIX         INX                     Get next character
                LDA     IN,X
.NOPREFIX       JSR     GET_EXPRES      Evaluate expression
                BNE     DIR_EXIT        An error occurred!
                LDY     CHAR            What prefix was used?
                LDA     HEXVAL
                CPY     #"#"
                BEQ     .LOW            Low byte only!
                CPY     #"/"
                BEQ     .HIGH           High byte only!
                JSR     SAVE_OBJ        Save low byte first
.HIGH           LDA     HEXVAL+1        Then save high byte
.LOW            JSR     SAVE_OBJ
                LDA     IN,X            More parameters
                CMP     #","
                BNE     DIR_EXIT        Nope!
                INX
                LDA     IN,X
                BNE     DIR_DA          Always taken!

;------------------------------------------------------------------------
; .EQ directive
;------------------------------------------------------------------------

DIR_EQ          LDA     PASS            What pass are we in?
                BNE     .RTS            Don't bother about .EQ during 2!
                JSR     GET_EXPRES      Evaluate expression
                BNE     DIR_OPE         An error occurred!
                LDA     UNDEF           Undefined labels are not allowed
                BNE     UNDEF_ERROR     Yes!

                LDA     IN              Was a label declared here?
                CMP     #"A"
                BCC     NOLABEL         Nope!
                CMP     #"Z"+1
                BCS     NOLABEL         Nope!
                LDY     #1              Point to label's value
                LDA     HEXVAL           and change it
                STA     (GLOBAL),Y
                INY
                LDA     HEXVAL+1
                STA     (GLOBAL),Y
.RTS            RTS

NOLABEL         LDA     #ERR_LBL        Exit with label error
                STA     ERROR
DIR_EXIT2       RTS

UNDEF_ERROR     LDA     #ERR_DEF        Exit with undefined label error
                STA     ERROR
                RTS

;------------------------------------------------------------------------
; .OR directive
;------------------------------------------------------------------------

DIR_OR          JSR     PRINT_OBJ       Print length of previous block
                JSR     GET_EXPRES      Evaluate expression
                BNE     FATAL_ERROR2    Can't accept errors here!
                STY     PC              Save PC and TA
                STA     PC+1
                BEQ     DIR_TA2         Always taken!

;------------------------------------------------------------------------
; .TA directive
;------------------------------------------------------------------------

DIR_TA          JSR     PRINT_OBJ       Print length of previous block
                JSR     GET_EXPRES      Evaluate expression
                BNE     DIR_OPE2        An error occurred!

DIR_TA2         STY     TA              Save target address
                STY     TA_BEGIN
                STA     TA+1
                STA     TA_BEGIN+1
                LDA     UNDEF           Undefined label used?
                BEQ     DIR_EXIT2       Nope! We're done now

DIR_UNDEF       LDA     #ERR_DEF        Can't allow undefined labels!
FATAL_ERROR     STA     ERROR
FATAL_ERROR2    JSR     CHCK_ERROR      Print the error
                JMP     GETLINE         And abort assembly! Fatal error

DIR_OPE2        JMP     DIR_OPE         Operand error

;------------------------------------------------------------------------
; Save byte to target
;------------------------------------------------------------------------

SAVE_OBJ        LDY     PASS            What pass?
                BEQ     .INCPC          Pass 1! Don't save anything
                PHA

                LDY     TA              See if below standard safe limit
                CPY     #DEF_OBJLOW
                LDA     TA+1
                SBC     /DEF_OBJLOW
                BCC     .USR            Below default safe range!
                CPY     LOMEM           See if it's above LOMEM now
                LDA     TA+1
                SBC     LOMEM+1
                BCC     .SAFE           We're in safe range!

.USR            CPY     USR_OBJLO       See if below user safe limit
                LDA     TA+1
                SBC     USR_OBJLO+1
                BCC     MEM_FULL        Yes! Exit with error
                CPY     USR_OBJHI       See if it's above user safe limit
                LDA     TA+1
                SBC     USR_OBJHI+1
                BCS     MEM_FULL        Yes! Exit with error

.SAFE           PLA
                LDY     #0
                STA     (TA),Y          Save byte
                INC     TA              Increment target address
                BNE     .INCPC          No carry!
                INC     TA+1

.INCPC          INC     PC              Increment PC
                BNE     .RTS            No carry!
                INC     PC+1
.RTS            RTS

MEM_FULL        LDA     #ERR_MEM        Exit with fatal memory error
                BNE     FATAL_ERROR      (ignoring the stack)

;------------------------------------------------------------------------
;  Expand the size of the symbol table by A
;  Presuming DEST points to current end flag ($00)
;  Warning END flag is not moved! Must be done by calling routine!
;------------------------------------------------------------------------

EXP_SYM         CLC                     See if there's enough room to
                ADC     DEST             expand
                STA     LENG
                LDA     #0
                TAY                     Point to first byte in new space
                ADC     DEST+1
                STA     LENG+1

                LDA     HIMEM           See if there's enough room
                CMP     LENG             for it
                LDA     HIMEM+1
                SBC     LENG+1
                BCC     MEM_FULL        Memory full!
                RTS

;------------------------------------------------------------------------
; Mnemonic pack macro. Packs 3 characters into 2 bytes to save memory
;------------------------------------------------------------------------

MNE             .MA     CHAR1,CHAR2,CHAR3
MNE_WORD        .SE     "]2"&%0001.1111
MNE_WORD        .SE     "]1"&%0001.1111*256+MNE_WORD
MNE_WORD        .SE     "]3"&%0001.1100*8*256+MNE_WORD
MNE_WORD        .SE     "]3"&%0000.0011*32+MNE_WORD
                .DR     MNE_WORD
                .EM

;------------------------------------------------------------------------
; Mnemonics tables
;------------------------------------------------------------------------

;------------------------------------------------------------------------
; Single byte, inherent operand instructions

MNEMONICS       .DA     #3              Record length
                .DA     SINGLES
                >MNE    B,R,K
                .DA     #$00
                >MNE    C,L,C
                .DA     #$18
                >MNE    C,L,D
                .DA     #$D8
                >MNE    C,L,I
                .DA     #$58
                >MNE    C,L,V
                .DA     #$B8
                >MNE    D,E,X
                .DA     #$CA
                >MNE    D,E,Y
                .DA     #$88
                >MNE    I,N,X
                .DA     #$E8
                >MNE    I,N,Y
                .DA     #$C8
                >MNE    N,O,P
                .DA     #$EA
                >MNE    P,H,A
                .DA     #$48
                >MNE    P,H,P
                .DA     #$08
                >MNE    P,L,A
                .DA     #$68
                >MNE    P,L,P
                .DA     #$28
                >MNE    R,T,I
                .DA     #$40
                >MNE    R,T,S
                .DA     #$60
                >MNE    S,E,C
                .DA     #$38
                >MNE    S,E,D
                .DA     #$F8
                >MNE    S,E,I
                .DA     #$78
                >MNE    T,A,X
                .DA     #$AA
                >MNE    T,A,Y
                .DA     #$A8
                >MNE    T,S,X
                .DA     #$BA
                >MNE    T,X,A
                .DA     #$8A
                >MNE    T,X,S
                .DA     #$9A
                >MNE    T,Y,A
                .DA     #$98
                .DO     PROCESSOR=$6502
                .EL
                >MNE    P,H,X
                .DA     #$DA
                >MNE    P,H,Y
                .DA     #$5A
                >MNE    P,L,X
                .DA     #$FA
                >MNE    P,L,Y
                .DA     #$7A
                .FI
                .DA     #0

;------------------------------------------------------------------------
; Branch instructions

                .DA     #3              Record length
                .DA     BRANCHES
                >MNE    B,C,C
                .DA     #$90
                >MNE    B,C,S
                .DA     #$B0
                >MNE    B,E,Q
                .DA     #$F0
                >MNE    B,M,I
                .DA     #$30
                >MNE    B,N,E
                .DA     #$D0
                >MNE    B,P,L
                .DA     #$10
                >MNE    B,V,C
                .DA     #$50
                >MNE    B,V,S
                .DA     #$70
                .DO     PROCESSOR=$6502
                .EL
                >MNE    B,R,A
                .DA     #$80
                .FI
                .DA     #0

;------------------------------------------------------------------------
; Definition of operand type flags
; Supported operand types are 1, unsupported types are 0 in flag field

OPE_IMM = %1000.0000
OPE_ZP  = %0100.0000
OPE_INH = %0010.0000
OPE_IND = %0001.0000
OPE_ZPX = %0000.1000
OPE_ABX = %0000.0100
OPE_ZPY = %0000.0010
OPE_ABY = %0000.0001

;------------------------------------------------------------------------
; Multiple operand type instructions

                .DA     #4              Record length
                .DA     MULTIPLES
                >MNE    A,D,C
                .DA     #$6D,#%1101.1101  Basic opcode and flag field
                >MNE    A,N,D
                .DA     #$2D,#%1101.1101
                >MNE    A,S,L
                .DA     #$0E,#%0110.1100

                >MNE    B,I,T
                .DO     PROCESSOR=$6502
                .DA     #$2C,#%0100.0000
                .EL
                .DA     #$2C,#%1100.1100
                .FI

                >MNE    C,M,P
                .DA     #$CD,#%1101.1101
                >MNE    C,P,X
                .DA     #$EC,#%1100.0000
                >MNE    C,P,Y
                .DA     #$CC,#%1100.0000

                >MNE    D,E,C
                .DO     PROCESSOR=$6502
                .DA     #$CE,#%0100.1100
                .EL
                .DA     #$CE,#%0110.1100
                .FI

                >MNE    E,O,R
                .DA     #$4D,#%1101.1101

                >MNE    I,N,C
                .DO     PROCESSOR=$6502
                .DA     #$EE,#%0100.1100
                .EL
                .DA     #$EE,#%0110.1100
                .FI

                >MNE    J,M,P
                .DO     PROCESSOR=$6502
                .DA     #$4C,#%0001.0000
                .EL
                .DA     #$4C,#%0001.0100
                .FI

                >MNE    J,S,R
                .DA     #$20,#%0000.0000
                >MNE    L,D,A
                .DA     #$AD,#%1101.1101
                >MNE    L,D,X
                .DA     #$AE,#%1100.0011
                >MNE    L,D,Y
                .DA     #$AC,#%1100.1100
                >MNE    L,S,R
                .DA     #$4E,#%0110.1100
                >MNE    O,R,A
                .DA     #$0D,#%1101.1101
                >MNE    R,O,L
                .DA     #$2E,#%0110.1100
                >MNE    R,O,R
                .DA     #$6E,#%0110.1100
                >MNE    S,B,C
                .DA     #$ED,#%1101.1101
                >MNE    S,T,A
                .DA     #$8D,#%0101.1101
                >MNE    S,T,X
                .DA     #$8E,#%0100.0010
                >MNE    S,T,Y
                .DA     #$8C,#%0100.1000

                .DO     PROCESSOR=$6502
                .EL
                >MNE    S,T,Z
                .DA     #$6C,#%0100.1100   Should be $9C, is exception!
                >MNE    T,R,B
                .DA     #$1C,#%0100.0000
                >MNE    T,S,B
                .DA     #$0C,#%0100.0000
                .FI
                .DA     #0,#0

;------------------------------------------------------------------------
; Decode mnemonic and operand
;------------------------------------------------------------------------

MNEMON          JSR     GET_MNEM        Get mnemonic from input and
                STA     PARM1            pack it
                JSR     GET_MNEM
                STA     PARM1+1
                JSR     GET_MNEM
                ASL                     Save bits 4, 3 and 2
                ASL                      together with 1st char
                ASL
                PHA
                AND     #%1110.0000
                ORA     PARM1
                STA     PARM1
                PLA
                ASL                     Save bits 1 and 0
                ASL                      together with 2nd char
                AND     #%0110.0000
                ORA     PARM1+1
                STA     PARM1+1
                JSR     IN_CR           Next char must be space or CR
                BEQ     .OK             It's a CR!
                CMP     #" "
                BNE     .MNE_ERR        Mnemonic error!

.OK             LDY     #0              Pointer in MNEMONICS table
.TABLE          LDA     MNEMONICS,Y     Get record length
                BEQ     .MNE_ERR        Mnemonic not found!
                STA     COUNT
                INY
                LDA     MNEMONICS,Y     Get handle routine
                STA     PARM2
                INY
                LDA     MNEMONICS,Y
                STA     PARM2+1
                INY

.LOOP           LDA     MNEMONICS,Y     Is it this mnemonic?
                BEQ     .TAB_END        End of table reached
                CMP     PARM1           Is it this one
                BNE     .NOT            Nope!
                LDA     MNEMONICS+1,Y
                CMP     PARM1+1
                BEQ     .FOUNDIT        Found mnemonic in table!
.NOT            CLC                     Add record length to Y
                TYA
                ADC     COUNT
                TAY
                BNE     .LOOP           Try next mnemonic!

.TAB_END        INY                     Point to begin of next table
                BNE     .TABLE          Do table loop again. Always!

.FOUNDIT        LDA     MNEMONICS+2,Y   Get basic opcode
                STA     PARM1
                LDA     MNEMONICS+3,Y   Get operand mask (if it's there)
                STA     PARM1+1

                JMP     (PARM2)         Execute handle routine

.MNE_ERR        LDA     #ERR_MNE        Exit with mnemonic error
                STA     ERROR
                RTS

;------------------------------------------------------------------------
; Get a MNEMONIC character, test it and prepare it for packing
;------------------------------------------------------------------------

GET_MNEM        LDA     IN,X            Get character
                INX                      point to next char
                AND     #%0001.1111     Mask irrelevant bits
GETM_RTS        RTS

;------------------------------------------------------------------------
; Single byte mnemonics
;------------------------------------------------------------------------

SINGLES         LDA     PARM1           Get opcode
                JMP     SAVE_OBJ        And send it to object

;------------------------------------------------------------------------
; Branch instructions
;------------------------------------------------------------------------

BRANCHES        JSR     SINGLES         Send opcode to object
                JSR     NNONSPC         Find next non space
                JSR     GET_EXPRES      Evaluate expression
                BNE     GETM_RTS        An error occurred
                LDA     UNDEF           Undefined label used?
                BNE     HEXVAL2OBJ      Yes! Can't verify range

                CLC                     Calculate displacement (+1!)
                LDA     HEXVAL
                SBC     PC
                STA     HEXVAL
                BPL     .NOBOR          No borrow needed
                INC     HEXVAL+1
.NOBOR          LDA     HEXVAL+1        See if high byte of displacement
                SBC     PC+1             is 0
                BEQ     HEXVAL2OBJ      OK!

.RANGE          JSR     RANGE_ERROR     Range error!
HEXVAL2OBJ      LDA     HEXVAL          Save displacement
                JMP     SAVE_OBJ

;------------------------------------------------------------------------
; Multiple choice operands
; Possible operands and their properties:
;
; ABS     is always available and is the basic opcode
;
; IMM     if it exists
;         $xC -> $x0      1100  0000
;         $xD -> $x9      1101  1001
;         $xE -> $x2      1110  0010
;         65C02: $CA -> $3A, $EA -> $1A
;
; ZP      if it exists
;         $xC -> $x4
;         $xD -> $x5
;         $xE -> $x6
;
; INH     if it exists
;         $xE -> $xA      1110 1010
;
; (IND,X) if it exists
;         $xD -> $x1
;
; (IND),Y if it exists (exists if (IND,X) exists)
;         $xD -> $x1 + $10
;
; ZP,X    if it exists
;         ZP + $10
;
; ZP,Y    if it exists
;         ZP + $10
;
; ABS,X   if it exists
;         ABS + $10
;
; ABS,Y   if it exists
;         IMM + $10
;------------------------------------------------------------------------

MULTIPLES       STX     SAVE_Y          Save current location in IN
                JSR     NNONSPC         Find operand
                BNE     .OP             Operand given!

;------------------------------------------------------------------------
; Inherent addressing mode

.NO_OP          LDA     PARM1+1         Does this opcode exist?
                AND     #OPE_INH
                BEQ     .OPE_ERROR      Nope!
                LDA     PARM1           Get basic opcode
                AND     #%1111.1011      translate to inherent opcode

                .DO     PROCESSOR=$6502
                .EL
                BPL     .LOW            Normal opcodes!
                EOR     #%1111.0000     Adapt INC and DEC for 65C02
                .FI

                BNE     .LOW            Always taken! Save to object

.OP             CMP     #";"            Is it a comment field?
                BEQ     .NO_OP          Yes! No operand given
                LDA     PARM1+1         Does inherent mode exist?
                AND     #OPE_INH
                BEQ     .IMM            Nope! Operand mandatory
                SEC                     See how far auto comment is
                TXA                      apart from mnemonic
                SBC     SAVE_Y
                CMP     #8
                BCS     .NO_OP          >8 spaces, treat as comment!

;------------------------------------------------------------------------
; Try to decode immediate addressing mode

.IMM            LDA     IN,X            Is it immediate mode?
                STA     CHAR
                CMP     #"#"
                BEQ     .DO_IMM         Yep!
                CMP     #"/"
                BNE     .IND            It's not! Try indirect mode
.DO_IMM         INX                     Point to expression
                LDA     PARM1+1         Does this mode exist?
                AND     #OPE_IMM
                BEQ     .OPE_ERROR      Nope!

                LDA     PARM1           Get basic opcode
                PHA                     Keep upper nibble
                AND     #%1111.0000
                STA     PARM1
                PLA
                AND     #%0000.1111     What category does it belong to?
                TAY
                LDA     #$09            In case it's $xD group
                CPY     #$0D
                BEQ     .IMM_OPC        It does!
                TYA
                AND     #%0000.0011     Translate $xC to $x0 & $xE to $x2
.IMM_OPC        ORA     PARM1           Combine opcode

                .DO     PROCESSOR=$6502
                .EL
                CMP     #$20            Is it BIT # ?
                BNE     .BIT_SKIP       Nope!
                LDA     #$89            This is the opcode for BIT #
.BIT_SKIP
                .FI

                JSR     SAVE_OBJ        Save opcode to object

                JSR     GET_EXPRES      Evaluate expression
                BNE     .LOW            An error occurred!
                LDY     CHAR            What prefix was used?
                LDA     HEXVAL
                CPY     #"#"            Low byte!
                BEQ     .LOW
                LDA     HEXVAL+1        Must be high byte now
.LOW            JMP     SAVE_OBJ

.OPE_ERROR      JMP     DIR_OPE         Exit with operand error

.RTS            RTS

;------------------------------------------------------------------------
; Try to decode indirect addressing mode

.IND            CMP     #"("            Can it be indirect mode?
                BNE     .ABSOLUTE       Nope!
                LDA     PARM1+1         Does this mode exist?
                AND     #OPE_IND
                BEQ     .OPE_ERROR      Nope!
                INX                     Evaluate expression
                JSR     GET_EXPRES
                BNE     .RTS            An error occurred!
                LDY     PARM1           Is it JMP IND?

                .DO     PROCESSOR=$6502
                LDA     #$6C             (Let's ignore closing brace
                CPY     #$4C              for memory's sake)
                BEQ     .SAVE_3         Yep! Save opcode and operand
                .EL
                CPY     #$4C            Is it JMP?
                BNE     .NOTJMP         Nope!
                LDA     #$6C            In case of JMP (IND)
                LDY     IN,X            Is it JMP (IND,X)?
                CPY     #","
                BNE     .SAVE_3         Can't be! Presume JMP (IND)
                LDA     #$7C            Let's presume it's JMP (IND,X)
                BNE     .SAVE_3         Always taken!
                .FI

.NOTJMP         LDA     UNDEF           Undefined label used?
                BNE     .IND_SKIP       Don't check range!
                LDA     HEXVAL+1        Address must fit on ZP
                BNE     .RANGE          Exit with range error!

.IND_SKIP       TYA                     Transform standard opcode
                AND     #%1111.0001      to (ZP,X) opcode
                STA     PARM1
                LDA     IN,X            What index mode is it?
                CMP     #","
                BNE     .IND_Y          Not ,X!
                INX
                LDA     IN,X            Next must be an X
                CMP     #"X"
                BNE     .OPE_ERROR      It's not!
                LDA     PARM1           Ignore closing brace
                BNE     .IND_SAVE       Always taken! Save to object

.IND_Y          INX                     Skip testing )

                .DO     PROCESSOR=$6502
                .EL
                LDA     IN,X            Is this a comma?
                CMP     #","
                BEQ     .NOTIND         Yes! So it's not (IND)!
                LDA     PARM1           Convert opcode to (IND)
                EOR     #%0001.0011
                BNE     .IND_SAVE       Always taken! Save to object
                .FI

.NOTIND         INX                     Skip testing comma
                LDA     IN,X
                CMP     #"Y"
                BNE     .OPE_ERROR
                LDA     PARM1           Get opcode
                ORA     #%0001.0000     Modify it to (ZP),Y

.IND_SAVE       JSR     SAVE_OBJ        Save opcode
                JMP     HEXVAL2OBJ       and save operand

.SAVE_3         JMP     .SAVE_ABS       Save 3 bytes

.RANGE          JMP     RANGE_ERROR     Exit with range error

;------------------------------------------------------------------------
; It can be either Absolute or ZP addressing now, with or without index

.ABSOLUTE       JSR     GET_EXPRES      Evaluate expression
                BNE     .RTS            An error occurred
                LDY     #0              In case no index mode
                LDA     IN,X            Indexed mode?
                CMP     #","
                BNE     .NOIDX          Nope!
                INX                     Get index character
                LDY     IN,X
.NOIDX          STY     INDEX           Save index mode (0, X or Y)
                LDA     UNDEF           Undefined or forward referenced?
                ORA     FORWARD
                ORA     HEXVAL+1        Is ZP possible?
                BNE     .NOZP           Nope! Use absolute instead

                LDA     PARM1+1         Get availability mask
                LDY     INDEX           Index mode used?
                BEQ     .ZP             Nope! Use ZP if it exists
                CPY     #"X"
                BEQ     .ZPX            Use ZP,X mode if it exists!
                CPY     #"Y"
                BEQ     .ZPY            Use ZP,Y mode if it exists!

.OPE_ERROR2     JMP     .OPE_ERROR      Exit with operand error

.ZP             AND     #OPE_ZP         Does this mode exist?
                BEQ     .NOZP           Nope! Try our luck with ABS
                LDA     PARM1           Get opcode
.SAVE_ZP        AND     #%1111.0111     Make it ZP mode
                JSR     SAVE_OBJ         and save it to object
                JMP     HEXVAL2OBJ      Save low byte of operand too

.ZPX            AND     #OPE_ZPX        Does this mode exist?
.ZPX1           BEQ     .NOZP           Nope! Try our luck with ABS
                LDA     PARM1           Modify opcode to indexed mode
                ORA     #%0001.0000
                BNE     .SAVE_ZP        Always taken!

.ZPY            AND     #OPE_ZPY        Does this mode exist?
                BNE     .ZPX1           Yes! Rest is like ZP,X

.NOZP           LDA     PARM1+1         Get availability mask
                LDY     INDEX           Index mode used?
                BEQ     .ABS            Nope! Absolute addressing!
                CPY     #"X"
                BEQ     .ABSX           Indexed with X!
                CPY     #"Y"
                BNE     .OPE_ERROR2     Wrong indexing mode!

.ABSY           AND     #OPE_ABY        Does this mode exist?
                BEQ     .OPE_ERROR2     Nope!
                LDA     PARM1           Modify operand
                AND     #%1111.1011
                CMP     #$AA            Is it LDX ABS,Y?
                BNE     .ABSX2          Nope! Rest is like ABS,X
                LDA     #$BE            Get proper opcode for LDX ABS,Y
                BNE     .SAVE_ABS       Always taken!

.ABSX           AND     #OPE_ABX        Does this mode exist?
                BEQ     .OPE_ERROR2     Nope!
                LDA     PARM1           Modify operand
.ABSX2          ORA     #%0001.0000

                .DO     PROCESSOR=$6502
                .EL
                CMP     #$7C            Is it STZ ABS?
                BNE     .SAVE_ABS       Nope
                LDA     #$9E            Correct opcode
                .FI

                BNE     .SAVE_ABS       And save the lot. Always taken!

.ABS            LDA     PARM1           Save unmodified opcode

                .DO     PROCESSOR=$6502
                .EL
                CMP     #$6C            Is it STZ ABS?
                BNE     .SAVE_ABS       Nope
                LDA     #$9C            Correct opcode
                .FI

.SAVE_ABS       JSR     SAVE_OBJ
                JSR     HEXVAL2OBJ      Followed by low and high byte
                LDA     HEXVAL+1
                JMP     SAVE_OBJ

;------------------------------------------------------------------------
; Find label in symbol table
; Upon exit:
; C=1 if not a legal label name was given
; Z=1 if label not found
; X will always point to end of label name + 1 in parse buffer
; Y will always be 0
; DEST will point at global label (or end of symbol table $00)
; DEST will point at local label (or end of local labels $FF)
;------------------------------------------------------------------------

FIND_LABEL      LDA     SYM_TABLE+1     Is symbol table valid?
                BEQ     .NOSYM          Nope!
                LDA     IN,X            Save first character of label
                STA     FIRST_CHAR
                CMP     #"."            Is it a local label?
                BEQ     FIND_LOCAL      Yes!

                CMP     #"A"            First character must be A-Z
                BCC     NO_LABEL        It's not!
                CMP     #"Z"+1
                BCS     NO_LABEL        It's not!

                JMP     FIND_GLOBAL     Find global label

.NOSYM          CLC                     Indicate label given
                JSR     UNDEF_ERROR     Exit with undefined label error
                LDA     #0              Make Z=0 meaning label not found
                RTS

NO_LABEL        SEC                     Indicate no label given
                RTS

;------------------------------------------------------------------------
; Find a local label in the symbol table
;------------------------------------------------------------------------

FIND_LOCAL      LDA     GLOBAL          Start looking from the begin of
                STA     DEST             the last assigned global label
                LDA     GLOBAL+1
                STA     DEST+1
                BEQ     NO_LABEL        No global label error!
                INX                     Next char must be a digit!
                LDA     IN,X
                CMP     #"0"
                BCC     NO_LABEL        It's not!
                CMP     #"9"+1
                BCS     NO_LABEL        It's not!

                JSR     GET_VAL         Get local label number
                LDA     HEXVAL
                STA     LL_NUM
                CMP     #100            Value must be below 100!
                BCS     NO_LABEL        It is not!
                LDA     HEXVAL+1
                BNE     NO_LABEL        It is not!

                LDY     #5-2            Point to start of local labels
.LOOP           INY                     Each record contains 2 bytes
                INY
                LDA     (DEST),Y        Could it be this one?
                CMP     #$FF
                BEQ     .DONE           Nope! End of record
                AND     #%0111.1111     Ignore label declared flag
                CMP     LL_NUM
                BNE     .LOOP           It's not this one!

                INY
                CLC                     Add label's offset to global
                LDA     GLOBAL_VAL       label's value
                ADC     (DEST),Y
                STA     HEXVAL
                LDA     GLOBAL_VAL+1
                ADC     #0
                STA     HEXVAL+1
                DEY

.DONE           CLC                     Add offset to pointer
                TYA
                ADC     DEST
                STA     DEST
                BCC     .NOCY           No carry
                INC     DEST+1
.NOCY           LDY     #0              Now clear Y (points to same addr)
                LDA     (DEST),Y        Set/clear label found flag
                CMP     #$FF            Z=1 if label not found
                CLC                     There was no error in label name
                RTS

;------------------------------------------------------------------------
; Find a global label in the symbol table
;------------------------------------------------------------------------

FIND_GLOBAL     LDA     SYM_TABLE       Start looking from the begin of
                STA     DEST             the symbol table
                LDA     SYM_TABLE+1
                STA     DEST+1

                STX     SAVE_Y          Save start of label name
.MARK_END       INX                     Find end of label we're looking
                LDA     IN,X             for
                CMP     #"."
                BEQ     .MARK_END       Not end yet!
                CMP     #"0"
                BCC     .FOUND_END      End!
                CMP     #"9"+1
                BCC     .MARK_END
                CMP     #"A"
                BCC     .FOUND_END      End!
                CMP     #"Z"+1
                BCC     .MARK_END
.FOUND_END      STX     DELIM           Character must be restored later!
                AND     #%0111.1111     Temporarily mark end of label
                STA     IN,X

.FND_GLOBAL     LDY     #0              Outer search loop
                LDA     (DEST),Y
                BEQ     .EXIT           End of symbol table reached!
                ORA     #%1000.0000     Ignore declared flag
                CMP     FIRST_CHAR      Should we bother looking further?
                BNE     .FAIL           Nope! Find next global label

                LDX     SAVE_Y          Point to begin of label again
                LDY     #4              Get pointer to label's name
                LDA     (DEST),Y         in source file
                STA     SRCE+1          (Going backwards leaves Y=3)
                DEY
                LDA     (DEST),Y
                STA     SRCE

.CMP_GLOBAL     LDA     IN,X            Compare strings
                BPL     .EOL            End of label reached!
                CMP     (SRCE),Y        Do they still match?
                BNE     .FAIL           Nope!
                INX
                INY
                BNE     .CMP_GLOBAL     Keep trying! Always taken

.EOL            LDA     (SRCE),Y        Is label in source also at end?
                BPL     .EXIT           Yes! (packed space or LineLeng)
                CMP     #":"
                BNE     .FAIL           Nope!

.EXIT           LDX     DELIM           Label found or at end of symtab
                LDA     IN,X            Restore end of label marker
                ORA     #%1000.0000
                STA     IN,X
                LDY     #0
                LDA     (DEST),Y        A will be 0 if label not found
                BEQ     .RTS            Label not found!
                EOR     #%1000.0000     Save label's assigned flag
                AND     #%1000.0000       (inverted!)
                AND     PASS            Remains 0 if pass 1
                STA     LABEL_FLAG
                INY
                LDA     (DEST),Y        Save label's value
                STA     HEXVAL
                INY
                LDA     (DEST),Y
                STA     HEXVAL+1
                LDY     #0              Restore pointer
                LDA     FIRST_CHAR       and first char (Z=0)
.RTS            CLC                     There was no error in label name
                RTS

.FAIL           LDY     #5              Find next label
.NXT_GLOBAL     LDA     (DEST),Y        Try to find $FF which signals
                INY                      the end of this label
                CMP     #$FF
                BNE     .NXT_GLOBAL     Not yet!
                TYA                     Add current offset to DEST
                CLC
                ADC     DEST
                STA     DEST
                BCC     .FND_GLOBAL     No carry needed!
                INC     DEST+1
                BNE     .FND_GLOBAL     Always taken!

;------------------------------------------------------------------------
; Add label
;------------------------------------------------------------------------

ADD_LABEL       JSR     FIND_LABEL      Find label in symbol table
                BCS     .LBL_ERR        It was not a valid label!
                BNE     .FOUND_IT       Found the label!
                LDA     FIRST_CHAR      What type of label is it?
                CMP     #"."
                BEQ     .ADD_LOCAL      It is a local label!

                LDA     #6              Make room for 6 bytes in the
                JSR     EXP_SYM          symbol table
                LDA     FIRST_CHAR      Save first character of label
                AND     #%0111.1111      in symbol table (clear declared
                STA     (DEST),Y          flag first)
                INY
                LDA     PC              Place current PC into symbol
                STA     (DEST),Y         table
                INY
                LDA     PC+1
                STA     (DEST),Y
                INY                     Place source pointer in symbol
                LDA     PNTR             table (where the label name
                STA     (DEST),Y          can be found)
                INY
                LDA     PNTR+1
                JSR     SAVE_ENDSYM     Save last 3 bytes of symbol table

.COPY_GLOBAL    LDA     DEST            Copy symbol table pointer
                STA     GLOBAL           to last global label
                LDA     DEST+1
                STA     GLOBAL+1
                LDY     #1              Copy last global label's value
                LDA     (DEST),Y
                STA     GLOBAL_VAL
                INY
                LDA     (DEST),Y
                STA     GLOBAL_VAL+1
                DEX

.BLANK          JSR     NEXT_IN         Find first blank or EOL
                BEQ     .RTS            EOL!
                CMP     #" "
                BNE     .BLANK          Try again!
.RTS            RTS

.LBL_ERR        JSR     NOLABEL         Exit with label error
                BNE     .BLANK          Always taken!

.FOUND_IT       LDA     PASS            What pass are we in?
                BEQ     .EXT_DEF        Pass 1! Extra definition

                LDA     (DEST),Y        Set declared flag in pass 2
                ORA     #%1000.0000
                STA     (DEST),Y

                LDA     FIRST_CHAR      What type was the label?
                CMP     #"."
                BNE     .COPY_GLOBAL    Copy last global label pointer
                RTS

.EXT_DEF        LDA     #ERR_EXT        Exit with extra definition error
                STA     ERROR
                RTS

.ADD_LOCAL      LDA     #2              Make room for 2 bytes in the
                JSR     EXP_SYM          symbol table
                LDA     LL_NUM          Save label number in symbol
                STA     (DEST),Y          table
                INY
                SEC                     Calculate offset from last global
                LDA     PC
                SBC     GLOBAL_VAL
                JSR     SAVE_ENDSYM     Save last 3 bytes of symbol table

                LDA     PC+1            Check offset range now
                SBC     GLOBAL_VAL+1
                BEQ     .RTS            Range is OK!

RANGE_ERROR     LDA     #ERR_RNG        Exit with range error
                STA     ERROR
                RTS

;------------------------------------------------------------------------
; Get next char from input buffer and compare it with CR
;------------------------------------------------------------------------

NEXT_IN_Y0      LDY     #0              Clear Y for different purposes
NEXT_IN         INX                     Well, let's do just that!
IN_CR           LDA     IN,X
                CMP     #CR
                RTS

;------------------------------------------------------------------------
; Save last 3 bytes to symbol table to save some code
;------------------------------------------------------------------------

SAVE_ENDSYM     STA     (DEST),Y        Save low byte of offset only
                INY
                LDA     #$FF            Save end of label marker
                STA     (DEST),Y
                INY
                LDA     #0              Save new end of symbol table flag
                STA     (DEST),Y
                RTS
